//------------------------------------------------------------------------------
// File: client_tools.cs
// This file contains the client methods for the tools
// Author: Matthew Rudge
//------------------------------------------------------------------------------

////////////////////////////////////////////////////////////////////////////////
//! Sets the mouse cursor when a tool has been selected. Default cursor is set
//! if tool is unknown.
//! \param %tool Tool selected
////////////////////////////////////////////////////////////////////////////////
function toolsSetCursor(%tool)
{
   switch(%tool) {
      case $INV_DYNAMITE:
      case $INV_HEALING:
         csUpdateCursor($CSC_REPAIR);
      case $INV_TOMBSTONE:
      case $INV_PLANTING:
      default:
         toolsResetCursor();
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Resets the cursor to default
////////////////////////////////////////////////////////////////////////////////
function toolsResetCursor()
{ 
   // Reset cursor
   csUpdateCursor($CSC_NONE);
}

////////////////////////////////////////////////////////////////////////////////
//! Executed when the tool is successfully selected on the client
//! \param %tool Tool that was selected
////////////////////////////////////////////////////////////////////////////////
function toolsOnSelect(%tool)
{
   // Set cursor for tool
   toolsSetCursor(%tool);
   
   switch(%tool) {
      // Create dynamite on server
      case $INV_DYNAMITE:
         commandToServer('OnDynamiteSelect');
      case $INV_HEALING:
      case $INV_TOMBSTONE:
      case $INV_PLANTING:
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Called while the tool command is active (in target state)
//! \param %tool Updated tool
////////////////////////////////////////////////////////////////////////////////
function toolsUpdateSelect(%tool)
{
   switch(%tool) {
      case $INV_DYNAMITE:
      // For healing do
      case $INV_HEALING:
         // Set target as current selection
         CmdStateTracker.setObjTarget(getSelectedObject(0));
      case $INV_TOMBSTONE:
      case $INV_PLANTING:
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Called when the tool use command was canceled
////////////////////////////////////////////////////////////////////////////////
function toolsOnCancel(%tool)
{
   switch(%tool) {
      case $INV_DYNAMITE:
         $DynamiteBuilding = false;
         %dynamite = PlayGui.removeObject();
         %client = ClientGroup.getObject(0);
         %ghostID = ServerConnection.getGhostID(%dynamite);
         serverCmdOnDynamiteCancel(%client, %ghostID);
      case $INV_HEALING:
      case $INV_TOMBSTONE:
      case $INV_PLANTING:
   }
   
   // Reset cursor
   toolsResetCursor();
   
   // Cancel inventory selection
   invCancelSelection();
}

////////////////////////////////////////////////////////////////////////////////
//! Called when the server has successfully selected and created a dynamite 
//! stick (ghost exists on this client)
//! \param %ghost Dynamite ghost id
////////////////////////////////////////////////////////////////////////////////
function clientCmdOnDynamiteSelect(%ghost)
{
   // Attach dynamite to cursor
   %dynamite = ServerConnection.resolveGhostID(%ghost);
   
   %placerList = gSelection.getSelectedGroup($SELECT_ID);
   if (%placerList.getSize() == 0)      
   {
      return;
   }
   %placer = %placerList.getId(0);
   PlayGui.attachObject(%placer, %dynamite);
   $ObjectToPlace = %dynamite;
}

////////////////////////////////////////////////////////////////////////////////
//! Called when the server has exploded the dynamite stick. Used for effects on
//! client.
//! \param %location Location of dynamite stick
////////////////////////////////////////////////////////////////////////////////
function clientCmdOnDynamiteExplode(%location)
{
   playSFXSound(audio_dynamiteexplode);
}

////////////////////////////////////////////////////////////////////////////////
//! Called when the server has placed the dynamite stick. Used for effects on
//! client.
//! \param %location Location of dynamite stick
////////////////////////////////////////////////////////////////////////////////
function clientCmdOnDynamitePlace(%location)
{
   playSFXSound(audio_dynamitesizzle);
}

////////////////////////////////////////////////////////////////////////////////
//! Called when healing has been used.
////////////////////////////////////////////////////////////////////////////////
function clientCmdOnHealUse()
{
   playSFXSound(audio_usehealthpack);
}

////////////////////////////////////////////////////////////////////////////////
//! Called when the server has successfully used a tool
//! \param %tool Tool used
////////////////////////////////////////////////////////////////////////////////
function clientCmdOnToolExecuteSuccess(%tool)
{
   // Detach object if attached to cursor
   if(PlayGui.hasObject()) {
      %object = PlayGui.removeObject();
      // Notify server right now that dynamite is immediately placed
      if(%tool == $INV_DYNAMITE) {
         commandToServer('OnDynamitePlace', ServerConnection.getGhostID(%object), %object.getPosition(), ServerConnection.getGhostID(getSelectedObject(0)));
      }
   }
   
   // Reset cursor
   toolsResetCursor();
   
   // Reset inventory
   invClientClearSelection();
}

// End client_tool.cs